﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.ComponentModel;
using CommonDataProvider;
using System.Collections.ObjectModel;
using System.Windows.Input;
using System.Windows.Data;

namespace Model_View_ViewModel_Enhanced
{
    public class ViewModel : INotifyPropertyChanged
    {
        public ViewModel()
        {
            EmployeeCollection = new ObservableCollection<Employee>();

            EmployeeListView = new ListCollectionView(EmployeeCollection);

            provider = new DataProvider();

            provider.IsSlow = true;

            Add = new CommandBinding(Commands.Add, Add_Executed);            
            Edit = new CommandBinding(Commands.Edit, Edit_Executed, Edit_CanExecute);
                        
            //We're using the default Application Commands for Delete!
            Delete = new CommandBinding(ApplicationCommands.Delete, Delete_Executed, Delete_CanExecute);

            Refresh = new CommandBinding(NavigationCommands.Refresh, Refresh_Executed);

            ChangeLastNameSort = new CommandBinding(MediaCommands.ChannelUp, ChangeLastNameSort_Executed);


            refreshWorker = new BackgroundWorker();
            refreshWorker.DoWork += new DoWorkEventHandler(refreshWorker_DoWork);
            refreshWorker.RunWorkerCompleted += new RunWorkerCompletedEventHandler(refreshWorker_RunWorkerCompleted);

            saveWorker = new BackgroundWorker();
            saveWorker.DoWork += new DoWorkEventHandler(saveWorker_DoWork);
            saveWorker.RunWorkerCompleted +=new RunWorkerCompletedEventHandler(saveWorker_RunWorkerCompleted);

            deleteWorker = new BackgroundWorker();
            deleteWorker.DoWork += new DoWorkEventHandler(deleteWorker_DoWork);
            deleteWorker.RunWorkerCompleted += new RunWorkerCompletedEventHandler(deleteWorker_RunWorkerCompleted);

            IsBusy = true;
            DoRefresh();            
        }

        
        private DataProvider provider;

        private ObservableCollection<Employee> EmployeeCollection;

        public ListCollectionView EmployeeListView { get; set; }

        #region Refresh
        BackgroundWorker refreshWorker;

        void refreshWorker_DoWork(object sender, DoWorkEventArgs e)
        {
            List<Employee> employees = provider.LoadAllEmployees();

            e.Result = employees;
        }

        void refreshWorker_RunWorkerCompleted(object sender, RunWorkerCompletedEventArgs e)
        {
            List<Employee> employees = e.Result as List<Employee>;

            EmployeeCollection.Clear();

            foreach (var employee in employees)
            {
                EmployeeCollection.Add(employee);
            }

            IsBusy = false;
        }

        public CommandBinding Refresh { get; private set; }
        private void Refresh_Executed(object sender, ExecutedRoutedEventArgs args)
        {
            if (IsBusy)
                return;

            IsBusy = true;

            DoRefresh();           
        }

        private void DoRefresh()
        {
            IsBusy = true;
            refreshWorker.RunWorkerAsync();
        }

        #endregion

        #region Add
        public CommandBinding Add { get; private set; }

        private void Add_Executed(object sender, ExecutedRoutedEventArgs args)
        {
            AddEventArgs addArgs = new AddEventArgs();


            FireAdd(addArgs);

            if (addArgs.AddConfirmed)
            {
                SaveToProvider(addArgs.NewEmployee);
            }
        }

        public event AddEventHandler AddRequested;

        private void FireAdd(AddEventArgs args)
        {           
            if (AddRequested != null)
            {
                AddRequested(this, args);
            }
        }

        #endregion

        #region Edit
        public CommandBinding Edit { get; private set; }


        private void Edit_CanExecute(object sender, CanExecuteRoutedEventArgs args)
        {
            args.CanExecute = (EmployeeListView.CurrentItem != null);
        }

        private void Edit_Executed(object sender, ExecutedRoutedEventArgs args)
        {
            //Extract the current element
            Employee selectedEmployee = EmployeeListView.CurrentItem as Employee;
            
            //Copy the employee
            Employee employeeCopy = Utilities.CreateCopy<Employee>(selectedEmployee);

            EditEventArgs editArgs = new EditEventArgs
            {
                Employee = employeeCopy
            };

            FireEdit(editArgs);

            if (editArgs.AcceptChanges)
            {
                SaveToProvider(editArgs.Employee);
            }
        }

        public event EditEventHandler EditRequested;

        private void FireEdit(EditEventArgs args)
        {           
            if (EditRequested != null)
            {
                EditRequested(this, args);
            }

        }

        #endregion

        #region Async Save
        private BackgroundWorker saveWorker;

        private void SaveToProvider(Employee employee)
        {
            IsBusy = true;
            saveWorker.RunWorkerAsync(employee);
        }

        void saveWorker_DoWork(object sender, DoWorkEventArgs e)
        {
            Employee employeeToSave = e.Argument as Employee;
            if (employeeToSave == null)
                return;
            provider.SaveEmployee(employeeToSave);
        }

        void saveWorker_RunWorkerCompleted(object sender, RunWorkerCompletedEventArgs e)
        {
            DoRefresh();   
        }
        #endregion

        #region Delete
        public CommandBinding Delete { get; private set; }
        private void Delete_CanExecute(object sender, CanExecuteRoutedEventArgs args)
        {
            args.CanExecute = (EmployeeListView.CurrentItem != null);
        }
        private void Delete_Executed(object sender, ExecutedRoutedEventArgs args)
        {
            IsBusy = true;

            Employee selectedEmployee = EmployeeListView.CurrentItem as Employee;

            IsBusy = true;
            deleteWorker.RunWorkerAsync(selectedEmployee);
        }
        private BackgroundWorker deleteWorker;

        void deleteWorker_DoWork(object sender, DoWorkEventArgs e)
        {
            Employee employeeToDelete = e.Argument as Employee;
            if (employeeToDelete == null)
                return;
            
            provider.DeleteEmployee(employeeToDelete);
        }

        void deleteWorker_RunWorkerCompleted(object sender, RunWorkerCompletedEventArgs e)
        {
            DoRefresh();
        }
        #endregion

        #region LastNameSort
        private SortDirection lastNameSort = SortDirection.Unsorted;
        public SortDirection LastNameSort
        {
            get
            {
                return lastNameSort;
            }
            set
            {
                lastNameSort = value;
                FirePropertyChanged("LastNameSort");
            }
        }

        public CommandBinding ChangeLastNameSort { get; private set; }
        private void ChangeLastNameSort_Executed(object sender, ExecutedRoutedEventArgs args)
        {
            switch (LastNameSort)
            {
                case SortDirection.Unsorted:
                    LastNameSort = SortDirection.Ascending;
                    break;
                case SortDirection.Ascending:
                    LastNameSort = SortDirection.Descending;
                    break;
                case SortDirection.Descending:
                    LastNameSort = SortDirection.Unsorted;
                    break;
                default:
                    break;
            }
            
            UpdateEmployeeListSort();
        }

        private void UpdateEmployeeListSort()
        {
            EmployeeListView.SortDescriptions.Clear();

            switch (LastNameSort)
            {
                case SortDirection.Ascending:
                    EmployeeListView.SortDescriptions.Add(new SortDescription("LastName", ListSortDirection.Ascending));
                    break;
                case SortDirection.Descending:
                    EmployeeListView.SortDescriptions.Add(new SortDescription("LastName", ListSortDirection.Descending));
                    break;
                default:
                    break;
            }
        }

        #endregion

        private bool isBusy;
        public bool IsBusy
        {
            get
            {
                return isBusy;
            }
            private set
            {
                isBusy = value;
                FirePropertyChanged("IsBusy");
            }
        }

        #region INotifyPropertyChanged Members

        private void FirePropertyChanged(string propertyName)
        {
            if (PropertyChanged != null)
            {
                PropertyChanged(this, new PropertyChangedEventArgs(propertyName));
            }
        }

        public event PropertyChangedEventHandler PropertyChanged;

        #endregion
    }

    public enum SortDirection
    {
        Unsorted,
        Ascending,
        Descending
    }
}
