using System;
using System.Collections.Generic;
using System.Text;
using System.Windows.Forms;
using System.Drawing;
using System.ComponentModel;

namespace DemoApplication
{
	/// <summary>
	/// A panel with a border.
	/// </summary>
	public class PanelWithColoredBorder : Panel
	{
		private Color m_objBorderColor = Color.Black;

		private Pen lazy_objBorderPen = null;
		private Pen BorderPen
		{
			get
			{
				if (lazy_objBorderPen == null)
					lazy_objBorderPen = new Pen( m_objBorderColor );
				return lazy_objBorderPen;
			}
		}

		/// <summary>
		/// Disposes the border pen.
		/// </summary>
		private void DisposeBorderPen()
		{
			if (lazy_objBorderPen != null)
			{
				lazy_objBorderPen.Dispose();
				lazy_objBorderPen = null;
			}
		}

		/// <summary>
		/// Releases the unmanaged resources used by the <see cref="T:System.Windows.Forms.Control"></see> and its child controls and optionally releases the managed resources.
		/// </summary>
		/// <param name="disposing">true to release both managed and unmanaged resources; false to release only unmanaged resources.</param>
		protected override void Dispose( bool disposing )
		{
			base.Dispose( disposing );
			if (disposing)
			{
				DisposeBorderPen();
			}
		}

		/// <summary>
		/// Raises the <see cref="E:System.Windows.Forms.Control.Paint"></see> event.
		/// </summary>
		/// <param name="e">A <see cref="T:System.Windows.Forms.PaintEventArgs"></see> that contains the event data.</param>
		protected override void OnPaint( PaintEventArgs e )
		{
			base.OnPaint( e );
//			e.Graphics.DrawRectangle( this.BorderPen, this.ClientRectangle ); // This doesn't work as expected
			Rectangle rect = this.ClientRectangle;
			--rect.Width;
			--rect.Height;
			e.Graphics.DrawRectangle(this.BorderPen, rect);
		}

		/// <summary>
		/// Gets or sets padding within the control.
		/// </summary>
		/// <value></value>
		/// <returns>A <see cref="T:System.Windows.Forms.Padding"></see> representing the control's internal spacing characteristics.</returns>
		public new Padding Padding
		{
			get { return base.Padding; }
			set
			{
				base.Padding = new Padding(
					Math.Max( value.Left, 2 ),
					Math.Max( value.Top, 2 ),
					Math.Max( value.Right, 2 ),
					Math.Max( value.Bottom, 2 ) );
			}
		}

		/// <summary>
		/// Gets or sets the color of the border.
		/// </summary>
		/// <value>The color of the border.</value>
		public Color BorderColor
		{
			get { return m_objBorderColor; }
			set
			{
				if (Color.Equals(m_objBorderColor,value))
					return;

				m_objBorderColor = value;
				DisposeBorderPen();
				this.Invalidate();
			}
		}

		/// <summary>
		/// Returns a value indicating whether the value of the <see cref="BorderColor"/> property should be serialized.
		/// </summary>
		/// <returns></returns>
		public bool ShouldSerializeBorderColor()
		{
			return (!Color.Equals( m_objBorderColor, Color.Black ));
		}

		/// <summary>
		/// Resets the value of the <see cref="BorderColor"/> property.
		/// </summary>
		/// <returns></returns>
		public void ResetBorderColor()
		{
			this.BorderColor = Color.Black;
		}

		/// <summary>
		/// Indicates the border style for the control.
		/// </summary>
		/// <value></value>
		[Browsable(false)]
		[DefaultValue(BorderStyle.FixedSingle)]
		public new BorderStyle BorderStyle
		{
			get { return BorderStyle.FixedSingle; }
			set { ;}
		}

		/// <summary>
		/// Initializes a new instance of the <see cref="PanelWithColoredBorder"/> class.
		/// </summary>
		public PanelWithColoredBorder()
		{
			base.BorderStyle = BorderStyle.None; // make sure the original border is hidden
			this.Padding = new Padding( 2 );
		}
	}
}
