#region Using directives
using System;
using System.Collections.Generic;
using System.Text;
using System.Windows.Forms;
using System.Drawing;
using System.Threading; 
#endregion

namespace bollmanncom.Tools
{
	/// <summary>
	/// Klasse, in der die SplashScreen-Funktionalitt gekapselt ist
	/// </summary>
	public static class SplashScreen
	{
        #region Declaration

        /// <summary>
        /// Betrag, um den die Durchsichtigkeit in jedem Schritt verringert wird
        /// </summary>
        private static double decreasingAmount = 0.02;

        /// <summary>
        /// Re-Entering-Flag fr decreaseOpacity
        /// </summary>
        private static bool decreasingOpacity = false;

        /// <summary>
        /// Timer fr das Ausblenden der Form
        /// </summary>
        private static System.Timers.Timer timer = new System.Timers.Timer();

        /// <summary>
        /// Texthintergrundfarbe fr Nachrichten auf dem SplashScreen
        /// </summary>
        private static Color backColor = Color.White;

        /// <summary>
        /// Eigentliche SplashScreen-Form
        /// </summary>
        private static SplashScreenForm splashScreenForm = new SplashScreenForm();

        /// <summary>
        /// Bereich, in dem Meldungen angezeigt werden knnen
        /// </summary>
        private static Rectangle messageRectangle = Rectangle.Empty;

        /// <summary>
        /// Soll der SplashScreen ausgefadet werden?
        /// </summary>
        private static bool fadeOutSplashScreen = true;

        /// <summary>
        /// Verzgerung in ms, die nach dem Anzeigen einer Nachricht eingelegt wird
        /// </summary>
        private static int messageDelay = 0;

        /// <summary>
        /// Hauptform
        /// </summary>
        private static Form mainForm = null;

        #endregion

        #region Construction

        /// <summary>
        /// Konstruktor
        /// </summary>
        static SplashScreen()
        {
            // Fading-Timer initialisieren
            timer.Interval = 10;
            timer.Enabled = false;
            timer.Elapsed += new System.Timers.ElapsedEventHandler(timer_Elapsed);
        }

        #endregion

        #region Properties

        /// <summary>
        /// SplashScreen aktiv?
        /// </summary>
        public static bool Active
        {
            get { return splashScreenForm.Visible; }
        }

        /// <summary>
        /// Hauptform
        /// </summary>
        public static Form MainForm
        {
            get { return mainForm; }
            private set 
            { 
                // Event abkoppeln
                if(mainForm != null)
                    mainForm.Load -= new EventHandler(mainForm_Load);

                mainForm = value;

                // Der Start der Form soll berwacht werden
                if(mainForm != null)
                    mainForm.Load += new EventHandler(mainForm_Load);
            }
        }

        /// <summary>
        /// Image, das als SplashScreen dargestellt wird
        /// </summary>
        public static Image SplashImage
        {
            get { return splashScreenForm.BackgroundImage; }
            set 
            {
                splashScreenForm.BackgroundImage = value;

                UpdateSplashScreenElements();
            }

        }

        /// <summary>
        /// Transparanzfarbe des SplashScreen-Images
        /// </summary>
        public static Color ImageTransparentColor
        {
            get { return splashScreenForm.TransparencyKey; }
            set 
            {
                // Hintergrundfarbe und TransparenceKey der Form auf den bergebenen Wert setzen
                splashScreenForm.BackColor = value;
                splashScreenForm.TransparencyKey = value; 
            }
        }

        /// <summary>
        /// Bereich, in dem Meldungen angezeigt werden knnen
        /// </summary>
        public static Rectangle MessageRectangle
        {
            get { return messageRectangle; }
            set 
            {
                messageRectangle = value;
                UpdateSplashScreenElements();
            }
        }

        /// <summary>
        /// Soll der SplashScreen ausgefadet werden?
        /// </summary>
        public static bool FadeOutSplashScreen
        {
            get { return fadeOutSplashScreen; }
            set { fadeOutSplashScreen = value; }
        }

        /// <summary>
        /// Verzgerung in ms, die nach dem Anzeigen einer Nachricht eingelegt wird
        /// </summary>
        public static int MessageDelay
        {
            get { return messageDelay; }
            set { messageDelay = value; }
        } 

        #endregion

        #region Methods

        /// <summary>
        /// SplashScreen-Elemente aktualisieren
        /// </summary>
        private static void UpdateSplashScreenElements()
        {
            if (SplashImage != null && !MessageRectangle.Equals(Rectangle.Empty))
            {
                // Bitmap vom SplashImage erstellen
                Bitmap b = new Bitmap(SplashImage);

                // Farbe des ersten Pixels ermitteln. Wir gehen davon aus,
                // da die Hintergrundfarbe des Textes einheitlich ist.
                backColor = b.GetPixel(MessageRectangle.Left, MessageRectangle.Top);

                // Bitmap wieder verwerfen
                b.Dispose();
            }

            // Hintergrundbild transparent machen
            Bitmap background = new Bitmap(splashScreenForm.BackgroundImage);
            background.MakeTransparent(ImageTransparentColor);
            splashScreenForm.BackgroundImage = background;
        }

        /// <summary>
        /// Anwendung starten
        /// </summary>
        /// <param name="mainForm">Hauptform</param>
        public static void RunApplication(Form mainForm)
        {
            // Hauptform setzen
            MainForm = mainForm;

            // Owner der Form setzen
            splashScreenForm.Owner = mainForm;

            // SplashScreen anzeigen
            splashScreenForm.Show();

            // Hauptform disablen
            MainForm.Enabled = false;

            // Application starten
            Application.Run(mainForm);
        }

        /// <summary>
        /// SplashScreen ausblenden
        /// </summary>
        public static void Hide()
        {
            // Timer fr das Ausblenden der Form anschmeissen
            timer.Start();
        }

        /// <summary>
        /// Durchsichtigkeit der SplashScreenForm verringern
        /// </summary>
        private static void decreaseOpacity()
        {
            // Re-Entering verhindern
            if (decreasingOpacity)
                return;

            // Re-Entering-Flag setzen
            decreasingOpacity = true;

            // Durchsichtigkeit herabsetzen
            double opacity = splashScreenForm.Opacity - decreasingAmount;

            // Wenn Durchsichtigkeit noch grer null, ...
            if (opacity >= 0)
                // ... Wert setzen
                splashScreenForm.Opacity = opacity;

            // Steuerung abgeben
            Application.DoEvents();

            // Re-Entering-Flag lschen
            decreasingOpacity = false;
        }

        #endregion

        #region Event handlers

        /// <summary>
        /// Fading-Timer wurde ausgelst
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        static void timer_Elapsed(object sender, System.Timers.ElapsedEventArgs e)
        {
            //Ggf. Hauptform enablen
            if(!MainForm.Enabled)
                splashScreenForm.BeginInvoke(new MethodInvoker(delegate () {
                    // Hauptform enablen
                    MainForm.Enabled = true;
                }));

            // Soll die Form gar nicht ausgeblendet werden oder ist Form bereits komplett durchsichtig oder Splashscreen gar nicht aktiv?
            if ( !FadeOutSplashScreen || splashScreenForm.Opacity < decreasingAmount || !Active)
            {
                // Timer stoppen
                timer.Stop();

                splashScreenForm.BeginInvoke(new MethodInvoker(delegate()
                {
                    // Form ausblenden
                    splashScreenForm.Hide();

                    // Hauptform aktivieren
                    MainForm.Activate();
                }));

                return;
            }

            // Durchsichtigkeit verringern
            splashScreenForm.BeginInvoke(new MethodInvoker(decreaseOpacity));
        }

        /// <summary>
        /// Hauptform wird dargestellt
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        static void mainForm_Load(object sender, EventArgs e)
        {
            // Hauptform disablen
            MainForm.Enabled = false;
        }

        /// <summary>
        /// Initialisierungsnachricht auf dem SplashScreen anzeigen
        /// </summary>
        /// <param name="message">Nachricht</param>
        public static void DrawMessage(string message)
        {
            // Wenn SplashScreen nicht aktiv, Methode verlassen
            if (!Active)
                return;

            // MessageRectangle darf nicht leer sein
            if (MessageRectangle.Equals(Rectangle.Empty))
                throw new Exception("MessageRectangle must not be empty!");

            // Graphics-Objekt erzeugen
            Graphics g = splashScreenForm.CreateGraphics();

            // Rechteckbereich auf der Form fllen
            g.FillRectangle(new SolidBrush(backColor), MessageRectangle);

            // Hintergrundfarbe invertieren --> Textfarbe
            Brush textBrush = new SolidBrush(Color.FromArgb(255 - backColor.R, 255 - backColor.G, 255 - backColor.B));

            // Nachricht zeichnen
            g.DrawString(message, splashScreenForm.Font, textBrush, MessageRectangle);

            // Steuerung abgeben
            Application.DoEvents();

            // Thread anhalten
            Thread.Sleep(MessageDelay);
        }

        #endregion
    }
}
